<?php
/**
 * This code is licensed under AGPLv3 license or Afterlogic Software License
 * if commercial version of the product was purchased.
 * For full statements of the licenses see LICENSE-AFTERLOGIC and LICENSE-AGPL3 files.
 */

namespace MailSo\Cache;

/**
 * @license https://www.gnu.org/licenses/agpl-3.0.html AGPL-3.0
 * @license https://afterlogic.com/products/common-licensing Afterlogic Software License
 * @copyright Copyright (c) 2019, Afterlogic Corp.
 *
 * @category MailSo
 * @package Cache
 */
class CacheClient
{
    /**
     * @var \MailSo\Cache\DriverInterface
     */
    private $oDriver;

    /**
     * @var string
     */
    private $sCacheIndex;

    /**
     * @access private
     */
    private function __construct()
    {
        $this->oDriver = null;
        $this->sCacheIndex = '';
    }

    /**
     * @return \MailSo\Cache\CacheClient
     */
    public static function NewInstance()
    {
        return new self();
    }

    /**
     * @param string $sKey
     * @param string $sValue
     *
     * @return bool
     */
    public function Set($sKey, $sValue)
    {
        return $this->oDriver ? $this->oDriver->Set($sKey.$this->sCacheIndex, $sValue) : false;
    }

    /**
     * @param string $sKey
     *
     * @return bool
     */
    public function SetTimer($sKey)
    {
        return $this->Set($sKey.'/TIMER', time());
    }

    /**
     * @param string $sKey
     * @param bool $bClearAfterGet = false
     *
     * @return string
     */
    public function get($sKey, $bClearAfterGet = false)
    {
        $sValue = '';

        if ($this->oDriver) {
            $sValue = $this->oDriver->get($sKey.$this->sCacheIndex);
        }

        if ($bClearAfterGet) {
            $this->Delete($sKey);
        }

        return $sValue;
    }

    /**
     * @param string $sKey
     *
     * @return int
     */
    public function GetTimer($sKey)
    {
        $iTimer = 0;
        $sValue = $this->get($sKey.'/TIMER');
        if (0 < strlen($sValue) && is_numeric($sValue)) {
            $iTimer = (int) $sValue;
        }

        return $iTimer;
    }

    /**
     * @param string $sKey
     *
     * @return \MailSo\Cache\CacheClient
     */
    public function Delete($sKey)
    {
        if ($this->oDriver) {
            $this->oDriver->Delete($sKey.$this->sCacheIndex);
        }

        return $this;
    }

    /**
     * @param \MailSo\Cache\DriverInterface $oDriver
     *
     * @return \MailSo\Cache\CacheClient
     */
    public function SetDriver(\MailSo\Cache\DriverInterface $oDriver)
    {
        $this->oDriver = $oDriver;

        return $this;
    }

    /**
     * @param int $iTimeToClearInHours = 24
     *
     * @return bool
     */
    public function gc($iTimeToClearInHours = 24)
    {
        return $this->oDriver ? $this->oDriver->gc($iTimeToClearInHours) : false;
    }

    /**
     * @return bool
     */
    public function IsInited()
    {
        return $this->oDriver instanceof \MailSo\Cache\DriverInterface;
    }

    /**
     * @param string $sCacheIndex
     *
     * @return \MailSo\Cache\CacheClient
     */
    public function SetCacheIndex($sCacheIndex)
    {
        $this->sCacheIndex = 0 < \strlen($sCacheIndex) ? "\x0".$sCacheIndex : '';

        return $this;
    }
}
