﻿using System;
using System.IO;
using System.Net;
using System.Text;
using System.Collections.Specialized;
using System.Collections.Generic;
using DotNetOpenAuth.OAuth2;
using Newtonsoft.Json;
using MailBee;
using MailBee.ImapMail;
using MailBee.SmtpMail;

class Sample
{
	// You get these at https://account.live.com/developers/applications/create
	const string clientID = "000000004C12XXXX";
	const string clientSecret = "VYXA3t02FHdjq-5JrQt5j2RzFoe5XXXX";

	// IMAP/SMTP, get e-mail address, offline access.
	const string LiveImapSmtpScopes = "wl.imap,wl.emails,wl.offline_access";
	const string LiveDesktopRedirectUri = "https://login.live.com/oauth20_desktop.srf";
	const string LiveAuthorizationUri = "https://login.live.com/oauth20_authorize.srf";
	const string LiveTokenEndpoint = "https://login.live.com/oauth20_token.srf";

	// Use this file as persistent storage of token data. The file is recreated if does not exist.
	const string tokenFile = "token.txt";

	// GetEmailAddress region's classes and methods are all about getting the e-mail address of the user.
	#region GetEmailAddress
	const string LiveGetProfileUri = "https://apis.live.net/v5.0/me?access_token=";

	protected class ExtendedMicrosoftClientUserData
	{
		public string FirstName { get; set; }
		public string Gender { get; set; }
		public string Id { get; set; }
		public string LastName { get; set; }
		public Uri Link { get; set; }
		public string Name { get; set; }
		public Emails Emails { get; set; }
	}

	protected class Emails
	{
		public string Preferred { get; set; }
		public string Account { get; set; }
		public string Personal { get; set; }
		public string Business { get; set; }
	}

	private static readonly string[] UriRfc3986CharsToEscape = new[] { "!", "*", "'", "(", ")" };
	private static string EscapeUriDataStringRfc3986(string value)
	{
		StringBuilder escaped = new StringBuilder(Uri.EscapeDataString(value));

		// Upgrade the escaping to RFC 3986, if necessary.
		for (int i = 0; i < UriRfc3986CharsToEscape.Length; i++)
		{
			escaped.Replace(UriRfc3986CharsToEscape[i], Uri.HexEscape(UriRfc3986CharsToEscape[i][0]));
		}

		// Return the fully-RFC3986-escaped string.
		return escaped.ToString();
	}

	// Inspired by http://answer.techwikihow.com/154458/getting-email-oauth-authentication-microsoft.html
	private static IDictionary<string, string> GetUserData(string accessToken)
	{
		ExtendedMicrosoftClientUserData graph;
		WebRequest request =
			WebRequest.Create(LiveGetProfileUri + EscapeUriDataStringRfc3986(accessToken));
		using (WebResponse response = request.GetResponse())
		{
			using (Stream responseStream = response.GetResponseStream())
			{
				using (StreamReader sr = new StreamReader(responseStream))
				{
					string data = sr.ReadToEnd();
					graph = JsonConvert.DeserializeObject<ExtendedMicrosoftClientUserData>(data);
				}
			}
		}

		Dictionary<string, string> userData = new Dictionary<string, string>();
		userData.Add("id", graph.Id);
		userData.Add("username", graph.Name);
		userData.Add("name", graph.Name);
		userData.Add("link", graph.Link == null ? null : graph.Link.AbsoluteUri);
		userData.Add("gender", graph.Gender);
		userData.Add("firstname", graph.FirstName);
		userData.Add("lastname", graph.LastName);
		userData.Add("email", graph.Emails.Preferred);
		return userData;
	}
	#endregion GetEmailAddress

	// Gets access and refresh token if grantedAccess is null, and returns the token data on success or null on failure.
	// Refreshes the access token using the refresh token if grantedAccess is not null, and returns the token data (also
	// modifying the input parameter grantedAccess) on successful renewal or null if the renewal is not required yet.
	public static AuthorizationState GetOrRefreshAccessToken(
		string clientID, string clientSecret, string key, AuthorizationState grantedAccess)
	{
		UserAgentClient consumer;

		bool getMode = (grantedAccess == null);

		StringDictionary parameters2 = new StringDictionary();
		parameters2.Add(OAuth2.RedirectUriKey, LiveDesktopRedirectUri);

		AuthorizationServerDescription server = new AuthorizationServerDescription();
		if (getMode)
		{
			server.AuthorizationEndpoint = new Uri(LiveAuthorizationUri);
		}
		server.TokenEndpoint = new Uri(LiveTokenEndpoint);
		server.ProtocolVersion = ProtocolVersion.V20;

		consumer = new UserAgentClient(server, clientID, clientSecret);

		// Not needed in our tests, uncomment the next line if for some reason it's required in your case.
		// consumer.ClientCredentialApplicator = ClientCredentialApplicator.PostParameter(clientSecret);

		if (getMode)
		{
			grantedAccess = new AuthorizationState(null);
			if (parameters2.ContainsKey(OAuth2.RedirectUriKey))
			{
				grantedAccess.Callback = new Uri(parameters2[OAuth2.RedirectUriKey]);
			}
			return (AuthorizationState)consumer.ProcessUserAuthorization(
				new Uri("http://example.com/?code=" + key), grantedAccess);
		}
		else
		{
			return consumer.RefreshAuthorization(grantedAccess, TimeSpan.FromMinutes(1)) ? grantedAccess : null;
		}
	}

	static void Main(string[] args)
	{
		AuthorizationState authState;
		string tokenData;
		bool mustSaveToken = true;
		if (File.Exists(tokenFile))
		{
			Console.WriteLine("Token data found, load the token from there");
			tokenData = File.ReadAllText(tokenFile);
			authState = JsonConvert.DeserializeObject<AuthorizationState>(tokenData);
			if (GetOrRefreshAccessToken(clientID, clientSecret, null, authState) != null)
			{
				// Note that GetOrRefreshAccessToken does modify authState object.
				// You don't need to actually save the returned value because authState parameter will already have
				// the updated data when this method completes.
				Console.WriteLine("Refreshing the token completed");
			}
			else
			{
				Console.WriteLine("Refreshing the token is not required yet");
				mustSaveToken = false;
			}
		}
		else
		{
			Console.WriteLine("No token data found, create a new token");

			OAuth2 myOAuth2 = new OAuth2(clientID, clientSecret);
			StringDictionary parameters1 = new StringDictionary();
			parameters1.Add(OAuth2.Scope, LiveImapSmtpScopes);
			parameters1.Add(OAuth2.RedirectUriKey, LiveDesktopRedirectUri);
			parameters1.Add(OAuth2.ResponseTypeKey, "code");

			System.Diagnostics.Process.Start(
				myOAuth2.AuthorizeToken(LiveAuthorizationUri, parameters1));

			// The page URL will be something like that:
			// https://login.live.com/oauth20_desktop.srf?code=c9afcb5f-fd66-4fec-49b4-9194f5a074d5&lc=1033
			// You need to copy/paste c9afcb5f-fd66-4fec-49b4-9194f5a074d5 value.
			Console.Write("Please enter the Authorization code: ");
			string key = Console.ReadLine().Trim();

			authState = GetOrRefreshAccessToken(clientID, clientSecret, key, null);
			if (authState == null)
			{
				Console.WriteLine("Can't get the access token");
				return;
			}
		}

		if (mustSaveToken)
		{
			tokenData = JsonConvert.SerializeObject(authState);
			File.WriteAllText(tokenFile, tokenData);
			Console.WriteLine("Token data saved");
		}

		// Set it if you already know the user's e-mail address.
		string userEmail = null;

		// Getting the user's e-mail address. Note that you don't need to do this on each run because the e-mail
		// address for the same user never changes. Remember the result once and then reuse it to avoid making
		// unnecessary network query.
		if (userEmail == null)
		{
			Console.WriteLine("Need to get the user's e-mail address");
			IDictionary<string, string> userData = GetUserData(authState.AccessToken);
			userEmail = userData["email"];
			Console.WriteLine("E-mail address is: " + userEmail);
		}

		// Get XOAuth key for IMAP.
		string xoauthKey = OAuth2.GetXOAuthKeyStatic(userEmail, authState.AccessToken);

		bool useImap = true;

		if (useImap)
		{
			Imap imp = new Imap();

			// Logging is not necessary but useful for debugging.
			imp.Log.Filename = "C:\\Temp\\log.txt";
			imp.Log.HidePasswords = false;
			imp.Log.Enabled = true;
			imp.Log.Clear();

			imp.Connect("imap-mail.outlook.com", 993);
			imp.Login(null, xoauthKey, AuthenticationMethods.SaslOAuth2, AuthenticationOptions.None, null);
			imp.SelectFolder("INBOX");

			Console.WriteLine(imp.MessageCount.ToString() + " message(s) in INBOX");

			imp.Disconnect();
		}
		else
		{
			Smtp mailer = new Smtp();
			mailer.SmtpServers.Add("smtp-mail.outlook.com", null, xoauthKey, AuthenticationMethods.SaslOAuth2);

			// Logging is not necessary but useful for debugging.
			mailer.Log.Filename = @"C:\Temp\log.txt";
			mailer.Log.HidePasswords = false;
			mailer.Log.Enabled = true;
			mailer.Log.Clear();

			mailer.From.Email = userEmail;
			mailer.To.Add(userEmail);
			mailer.Subject = "empty email to myself";
			mailer.Send();
			Console.WriteLine("E-mail sent");
		}
	}
}